"use client"

import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Dialog, DialogContent, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import AppLayout from "@/layouts/app-layout"
import { dashboard } from "@/routes"
import { type BreadcrumbItem } from "@/types"
import { Head, useForm } from "@inertiajs/react"
import { Edit, Eye, Plus, Trash2 } from "lucide-react"
import { useState, useMemo } from "react"

interface Makanan {
  id: number
  nama: string
  kategori: string
  stok: number
  harga: number
}

interface Props {
  makanan: Makanan[]
}

const breadcrumbs: BreadcrumbItem[] = [
  { title: "Stok Makanan", href: dashboard().url },
]

export default function Index({ makanan }: Props) {
  const { data, setData, post, put, delete: destroy, processing, reset, errors } =
    useForm({
      nama: "",
      kategori: "Makanan",
      stok: 0,
      harga: 0,
    })

  const [search, setSearch] = useState("")
  const [openAdd, setOpenAdd] = useState(false)
  const [openEdit, setOpenEdit] = useState(false)
  const [openDetail, setOpenDetail] = useState(false)
  const [selected, setSelected] = useState<Makanan | null>(null)

  const filteredData = useMemo(() => {
    return makanan.filter((item) =>
      item.nama.toLowerCase().includes(search.toLowerCase())
    )
  }, [search, makanan])

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Stok Makanan" />
      <Card className="m-4">
        <CardHeader className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-2">
          <CardTitle className="text-2xl font-bold">Manajemen Stok</CardTitle>
          <div className="flex items-center gap-2">
            <Input
              placeholder="Cari makanan..."
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              className="w-48"
            />
            <Button
              onClick={() => {
                reset()
                setData("kategori", "Makanan")
                setOpenAdd(true)
              }}
            >
              <Plus className="w-4 h-4 mr-1" /> Tambah
            </Button>
          </div>
        </CardHeader>

        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Nama</TableHead>
                <TableHead>Kategori</TableHead>
                <TableHead>Stok</TableHead>
                <TableHead>Harga</TableHead>
                <TableHead>Aksi</TableHead>
              </TableRow>
            </TableHeader>
            {filteredData.length > 0 ? (
              <TableBody>
                {filteredData.map((item) => (
                  <TableRow key={item.id}>
                    <TableCell>{item.nama}</TableCell>
                    <TableCell>
                      <Badge
                        className={
                          item.kategori === "Makanan"
                            ? "bg-green-100 text-green-800"
                            : "bg-blue-100 text-blue-800"
                        }
                      >
                        {item.kategori}
                      </Badge>
                    </TableCell>
                    <TableCell>{item.stok}</TableCell>
                    <TableCell>Rp {item.harga.toLocaleString()}</TableCell>
                    <TableCell>
                      <div className="flex gap-2">
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            setSelected(item)
                            setOpenDetail(true)
                          }}
                        >
                          <Eye className="w-4 h-4" />
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => {
                            setSelected(item)
                            setData("nama", item.nama)
                            setData("kategori", item.kategori)
                            setData("stok", item.stok)
                            setData("harga", item.harga)
                            setOpenEdit(true)
                          }}
                        >
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button
                          size="sm"
                          variant="destructive"
                          onClick={() => {
                            if (confirm("Yakin ingin menghapus data ini?")) {
                              destroy(`/makanan/${item.id}`, {
                                preserveScroll: true,
                                onSuccess: () => {
                                  reset()
                                  alert("Data berhasil dihapus")
                                },
                              })
                            }
                          }}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            ) : (
              <TableRow>
                <TableCell colSpan={5} className="py-6 text-center text-muted-foreground">
                  {search ? "Data tidak ditemukan" : "Belum ada data makanan"}
                </TableCell>
              </TableRow>
            )}
          </Table>
        </CardContent>
      </Card>

      {/* ➕ Modal Tambah */}
      <Dialog open={openAdd} onOpenChange={setOpenAdd}>
        <DialogContent className="max-w-lg">
          <DialogHeader><DialogTitle>Tambah Makanan</DialogTitle></DialogHeader>
          <form
            onSubmit={(e) => {
              e.preventDefault()
              post("/makanan", {
                preserveScroll: true,
                onSuccess: () => {
                  reset()
                  setOpenAdd(false)
                  alert("Data berhasil disimpan")
                },
              })
            }}
          >
            <div className="space-y-4">
              <div>
                <Label>Nama</Label>
                <Input
                  value={data.nama}
                  onChange={(e) => setData("nama", e.target.value)}
                />
                {errors.nama && <p className="text-red-500 text-sm">{errors.nama}</p>}
              </div>

              <div>
                <Label>Kategori</Label>
                <Select
                  value={data.kategori}
                  onValueChange={(v) => setData("kategori", v)}
                >
                  <SelectTrigger><SelectValue /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="Makanan">Makanan</SelectItem>
                    <SelectItem value="Minuman">Minuman</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Stok</Label>
                <Input
                  type="number"
                  value={data.stok}
                  onChange={(e) => setData("stok", parseInt(e.target.value))}
                />
              </div>

              <div>
                <Label>Harga</Label>
                <Input
                  type="number"
                  value={data.harga}
                  onChange={(e) => setData("harga", parseInt(e.target.value))}
                />
              </div>

              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => setOpenAdd(false)}>
                  Batal
                </Button>
                <Button type="submit" disabled={processing}>Simpan</Button>
              </DialogFooter>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      {/* ✏️ Modal Edit */}
      <Dialog open={openEdit} onOpenChange={setOpenEdit}>
        <DialogContent className="max-w-lg">
          <DialogHeader><DialogTitle>Edit Makanan</DialogTitle></DialogHeader>
          <form
            onSubmit={(e) => {
              e.preventDefault()
              if (!selected) return
              put(`/makanan/${selected.id}`, {
                preserveScroll: true,
                onSuccess: () => {
                  reset()
                  setOpenEdit(false)
                  alert("Data berhasil diupdate")
                },
              })
            }}
          >
            <div className="space-y-4">
              <div>
                <Label>Nama</Label>
                <Input
                  value={data.nama}
                  onChange={(e) => setData("nama", e.target.value)}
                />
              </div>

              <div>
                <Label>Kategori</Label>
                <Select
                  value={data.kategori}
                  onValueChange={(v) => setData("kategori", v)}
                >
                  <SelectTrigger><SelectValue /></SelectTrigger>
                  <SelectContent>
                    <SelectItem value="Makanan">Makanan</SelectItem>
                    <SelectItem value="Minuman">Minuman</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Stok</Label>
                <Input
                  type="number"
                  value={data.stok}
                  onChange={(e) => setData("stok", parseInt(e.target.value))}
                />
              </div>

              <div>
                <Label>Harga</Label>
                <Input
                  type="number"
                  value={data.harga}
                  onChange={(e) => setData("harga", parseInt(e.target.value))}
                />
              </div>

              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => setOpenEdit(false)}>
                  Batal
                </Button>
                <Button type="submit" disabled={processing}>Update</Button>
              </DialogFooter>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      {/* 👁️ Modal Detail */}
      <Dialog open={openDetail} onOpenChange={setOpenDetail}>
        <DialogContent>
          <DialogHeader><DialogTitle>Detail Makanan</DialogTitle></DialogHeader>
          {selected && (
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="font-medium">Nama</span>
                <span>{selected.nama}</span>
              </div>
              <div className="flex justify-between">
                <span className="font-medium">Kategori</span>
                <span>{selected.kategori}</span>
              </div>
              <div className="flex justify-between">
                <span className="font-medium">Stok</span>
                <span>{selected.stok}</span>
              </div>
              <div className="flex justify-between">
                <span className="font-medium">Harga</span>
                <span>Rp {selected.harga.toLocaleString()}</span>
              </div>
            </div>
          )}
          <DialogFooter>
            <Button variant="outline" onClick={() => setOpenDetail(false)}>Tutup</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AppLayout>
  )
}
