<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Billing;
use App\Models\EspRelay;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class BillingController extends Controller
{
    /**
     * Start a new billing session
     */
    public function start(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'device_id' => 'required|string',
                'pin' => 'required|integer',
                'nama_pelanggan' => 'required|string',
                'mode' => 'required|in:bebas,timer',
                'tarif_perjam' => 'required|numeric|min:0',
                'promo_id' => 'nullable|exists:promos,id',
                'durasi' => 'nullable|string', // Required for timer mode (HH:MM:SS format)
                'paket_id'      => 'nullable|integer',
                'user_id'        => 'required|integer', // Required from frontend
            ]);

            // Find the ESP relay by device_id and pin
            $espRelay = EspRelay::where('device_id', $validated['device_id'])
                ->where('pin', $validated['pin'])
                ->first();

            if (!$espRelay) {
                return response()->json([
                    'success' => false,
                    'message' => 'ESP Relay not found for device and pin',
                ], 404);
            }

            // Check if there's already an active billing for this relay
            $existingBilling = Billing::where('esp_relay_id', $espRelay->id)
                ->where('status', 'aktif')
                ->first();

            if ($existingBilling) {
                return response()->json([
                    'success' => false,
                    'message' => 'There is already an active billing session for this relay',
                ], 400);
            }

            // Validate duration for timer mode
            $durasi = null;
            if ($validated['mode'] === 'timer') {
                if (!isset($validated['durasi']) || empty($validated['durasi'])) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Duration is required for timer mode',
                    ], 400);
                }
                $durasi = $validated['durasi'];
            }

            // Hitung total awal otomatis jika mode timer
            $totalBiayaAwal = 0;
            if ($validated['mode'] === 'timer' && isset($validated['durasi'])) {
                $parts = explode(':', $validated['durasi']);
                $jam = (int) $parts[0];
                $menit = (int) $parts[1];
                $durasiJam = $jam + ($menit / 60);
                $totalBiayaAwal = $durasiJam * (float) $validated['tarif_perjam'];
            }

            // Create new billing record
            $billing = Billing::create([
                'esp_relay_id' => $espRelay->id,
                'promo_id' => $validated['promo_id'],
                'nama_pelanggan' => $validated['nama_pelanggan'],
                'mode' => $validated['mode'],
                'status' => 'aktif',
                'tarif_perjam' => $validated['tarif_perjam'],
                'durasi' => $durasi, // Set duration for timer mode
                'paket_id'      => $validated['paket_id'] ?? null,
                'waktu_mulai' => Carbon::now(),
                'user_id'        => $validated['user_id'], // Use user_id from frontend
                'total_biaya' => $totalBiayaAwal, // tambahkan baris ini
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Billing session started successfully',
                'data' => [
                    'billing_id' => $billing->id,
                    'esp_relay_id' => $billing->esp_relay_id,
                    'nama_pelanggan' => $billing->nama_pelanggan,
                    'mode' => $billing->mode,
                    'tarif_perjam' => $billing->tarif_perjam,
                    'durasi' => $billing->durasi,
                    'waktu_mulai' => $billing->waktu_mulai->toISOString(),
                    'paket_id' => $billing->paket_id,
                    'user_id' => $billing->user_id,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to start billing session: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Stop billing session and calculate final costs
     */
    public function stop(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'device_id' => 'required|string',
                'pin' => 'required|integer',
                'total_biaya' => 'required|numeric|min:0',
                'durasi' => 'nullable|string', // Now accepts time format HH:MM:SS
            ]);

            // Find the ESP relay by device_id and pin
            $espRelay = EspRelay::where('device_id', $validated['device_id'])
                ->where('pin', $validated['pin'])
                ->first();

            if (!$espRelay) {
                return response()->json([
                    'success' => false,
                    'message' => 'ESP Relay not found for device and pin',
                ], 404);
            }

            // Find active billing for this relay
            $billing = Billing::where('esp_relay_id', $espRelay->id)
                ->where('status', 'aktif')
                ->first();

            if (!$billing) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active billing session found for this relay',
                ], 404);
            }

            // Calculate actual duration for both modes
            $waktuMulai = Carbon::parse($billing->waktu_mulai);
            $waktuSekarang = Carbon::now();
            $totalSeconds = $waktuSekarang->diffInSeconds($waktuMulai);

            // Convert actual elapsed time to HH:MM:SS format
            $hours = floor($totalSeconds / 3600);
            $minutes = floor(($totalSeconds % 3600) / 60);
            $seconds = $totalSeconds % 60;
            $durasiAktual = sprintf('%02d:%02d:%02d', $hours, $minutes, $seconds);

            // Use provided duration if available, otherwise use calculated actual duration
            $durasi = $validated['durasi'] ?? $durasiAktual;

            // Update billing record
            $billing->update([
                'status' => 'selesai',
                'total_biaya' => $validated['total_biaya'],
                'durasi' => $durasi,
                'waktu_selesai' => Carbon::now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Billing session stopped successfully',
                'data' => [
                    'billing_id' => $billing->id,
                    'nama_pelanggan' => $billing->nama_pelanggan,
                    'mode' => $billing->mode,
                    'total_biaya' => $billing->total_biaya,
                    'durasi' => $billing->durasi,
                    'waktu_mulai' => $billing->waktu_mulai->toISOString(),
                    'waktu_selesai' => $billing->waktu_selesai->toISOString(),
                    'paket_id' => $billing->paket_id,
                    'user_id' => $billing->user_id,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to stop billing session: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function cancel(Request $request): JsonResponse
{
    try {
        $validated = $request->validate([
            'device_id' => 'required|string',
            'pin' => 'required|integer',
        ]);

        // Cari relay berdasarkan device dan pin
        $espRelay = EspRelay::where('device_id', $validated['device_id'])
            ->where('pin', $validated['pin'])
            ->first();

        if (!$espRelay) {
            return response()->json([
                'success' => false,
                'message' => 'ESP Relay tidak ditemukan untuk device dan pin ini',
            ], 404);
        }

        // Cari billing aktif
        $billing = Billing::where('esp_relay_id', $espRelay->id)
            ->where('status', 'aktif')
            ->first();

        if (!$billing) {
            return response()->json([
                'success' => false,
                'message' => 'Tidak ada sesi billing aktif untuk relay ini',
            ], 404);
        }

        // Cek apakah kolom status mendukung nilai 'dibatalkan'
        $allowedStatuses = \DB::select("SHOW COLUMNS FROM billings LIKE 'status'");
        $enumValues = [];
        if (!empty($allowedStatuses)) {
            preg_match("/^enum\((.*)\)$/", $allowedStatuses[0]->Type, $matches);
            if (!empty($matches[1])) {
                $enumValues = str_getcsv(str_replace("'", "", $matches[1]));
            }
        }

        // Tentukan status pembatalan yang valid
        $cancelStatus = in_array('dibatalkan', $enumValues) ? 'dibatalkan' : 'selesai';

        // Update billing jadi dibatalkan
        $billing->update([
            'status' => $cancelStatus,
            'total_biaya' => 0,
            'total_setelah_promo' => 0,
            'waktu_selesai' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Billing berhasil dibatalkan',
            'data' => [
                'billing_id' => $billing->id,
                'status' => $billing->status,
                'waktu_selesai' => $billing->waktu_selesai,
            ],
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Gagal membatalkan billing: ' . $e->getMessage(),
        ], 500);
    }
}

public function tambahWaktu(Request $request): JsonResponse
{
    try {
        $validated = [
            'device_id' => $request->device_id,
            'pin' => $request->pin,
            'tambahan_durasi' => $request->tambahan_durasi ?? $request->tambahanDurasi ?? $request->durasi,
        ];

        $espRelay = \App\Models\EspRelay::where('device_id', $validated['device_id'])
            ->where('pin', $validated['pin'])
            ->first();

        if (!$espRelay) {
            return response()->json(['success' => false, 'message' => 'ESP Relay tidak ditemukan'], 404);
        }

        $billing = \App\Models\Billing::where('esp_relay_id', $espRelay->id)
            ->whereIn('status', ['aktif', 'on', 'running'])
            ->first();

        if (!$billing) {
            return response()->json(['success' => false, 'message' => 'Tidak ada sesi billing aktif'], 404);
        }

        $durasiLama = explode(':', $billing->durasi ?? '00:00:00');
        $tambahan = explode(':', $validated['tambahan_durasi'] ?? '00:00:00');

        while (count($durasiLama) < 3) array_push($durasiLama, 0);
        while (count($tambahan) < 3) array_push($tambahan, 0);

        $totalDetik = ($durasiLama[0]*3600 + $durasiLama[1]*60 + $durasiLama[2]) +
                      ($tambahan[0]*3600 + $tambahan[1]*60 + $tambahan[2]);

        $jamBaru = floor($totalDetik / 3600);
        $menitBaru = floor(($totalDetik % 3600) / 60);
        $detikBaru = $totalDetik % 60;

        $durasiBaru = sprintf('%02d:%02d:%02d', $jamBaru, $menitBaru, $detikBaru);
        $totalJam = $totalDetik / 3600;
        $totalBiaya = $billing->tarif_perjam * $totalJam;

        $billing->update([
            'durasi' => $durasiBaru,
            'total_biaya' => $totalBiaya,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Durasi billing berhasil diperpanjang',
            'data' => ['durasi_baru' => $durasiBaru, 'total_biaya' => $totalBiaya]
        ]);
    } catch (\Throwable $e) {
        return response()->json([
            'success' => false,
            'message' => 'Gagal menambah waktu: ' . $e->getMessage(),
        ], 500);
    }
}

public function move(Request $request): JsonResponse
{
    try {
        $validated = $request->validate([
            'device_id_asal' => 'required|string',
            'port_asal' => 'required|string',
            'device_id_tujuan' => 'required|string',
            'port_tujuan' => 'required|string',
        ]);

        $deviceAsal = strtoupper($validated['device_id_asal']);
        $deviceTujuan = strtoupper($validated['device_id_tujuan']);
        $portAsal = strtoupper($validated['port_asal']);
        $portTujuan = strtoupper($validated['port_tujuan']);

        // Pastikan prefix "PORT " ada
        if (!str_starts_with($portAsal, 'PORT ')) $portAsal = 'PORT ' . $portAsal;
        if (!str_starts_with($portTujuan, 'PORT ')) $portTujuan = 'PORT ' . $portTujuan;

        // Cari relay asal dan tujuan berdasarkan device_id + nama_relay
        $relayAsal = \DB::table('esp_relay_logs')
            ->where('device_id', $deviceAsal)
            ->where('nama_relay', $portAsal)
            ->first();

        $relayTujuan = \DB::table('esp_relay_logs')
            ->where('device_id', $deviceTujuan)
            ->where('nama_relay', $portTujuan)
            ->first();

        if (!$relayAsal || !$relayTujuan) {
            return response()->json([
                'success' => false,
                'message' => 'Port asal atau tujuan tidak ditemukan'
            ], 404);
        }

        // Cek billing aktif di port asal
        $billing = \App\Models\Billing::where('esp_relay_id', $relayAsal->id)
            ->where('status', 'aktif')
            ->first();

        if (!$billing) {
            return response()->json([
                'success' => false,
                'message' => 'Tidak ada billing aktif di port asal'
            ], 404);
        }

        // Cek apakah port tujuan sedang digunakan
        $billingTujuanAktif = \App\Models\Billing::where('esp_relay_id', $relayTujuan->id)
            ->where('status', 'aktif')
            ->first();

        if ($billingTujuanAktif) {
            return response()->json([
                'success' => false,
                'message' => 'Port tujuan sedang digunakan'
            ], 400);
        }

        // Update billing ke port tujuan
        $billing->update([
            'esp_relay_id' => $relayTujuan->id,
            'updated_at' => now(),
        ]);

        // Update status port
        \DB::table('esp_relay_logs')->where('id', $relayAsal->id)->update(['status' => 0]);
        \DB::table('esp_relay_logs')->where('id', $relayTujuan->id)->update(['status' => 1]);

        return response()->json([
            'success' => true,
            'message' => 'Berhasil pindah room!',
        ]);
    } catch (\Throwable $e) {
        return response()->json([
            'success' => false,
            'message' => 'Gagal pindah room: ' . $e->getMessage(),
        ], 500);
    }
}
    /**
     * Get active billing for a specific relay
     */
    public function getActiveBilling(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'device_id' => 'required|string',
                'pin' => 'required|integer',
            ]);

            // Find the ESP relay by device_id and pin
            $espRelay = EspRelay::where('device_id', $validated['device_id'])
                ->where('pin', $validated['pin'])
                ->first();

            if (!$espRelay) {
                return response()->json([
                    'success' => false,
                    'message' => 'ESP Relay not found for device and pin',
                ], 404);
            }

            // Find active billing for this relay
            $billing = Billing::with(['promo'])
                ->where('esp_relay_id', $espRelay->id)
                ->where('status', 'aktif')
                ->first();

            if (!$billing) {
                return response()->json([
                    'success' => true,
                    'message' => 'No active billing session found',
                    'data' => null,
                ]);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'billing_id' => $billing->id,
                    'nama_pelanggan' => $billing->nama_pelanggan,
                    'mode' => $billing->mode,
                    'tarif_perjam' => $billing->tarif_perjam,
                    'waktu_mulai' => $billing->waktu_mulai->toISOString(),
                    'promo' => $billing->promo ? [
                        'id' => $billing->promo->id,
                        'name' => $billing->promo->name,
                        'code' => $billing->promo->code,
                    ] : null,
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get active billing: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Check and auto-stop expired timed billings
     */
    public function checkExpiredTimedBillings(): JsonResponse
    {
        try {
            // Get all active timer mode billings
            $expiredBillings = Billing::with(['espRelay'])
                ->where('status', 'aktif')
                ->where('mode', 'timer')
                ->whereNotNull('durasi')
                ->get()
                ->filter(function ($billing) {
                    // Check if billing has expired
                    $waktuMulai = Carbon::parse($billing->waktu_mulai);
                    $durasi = $billing->durasi; // Already in HH:MM:SS format

                    // Parse duration to add to start time
                    $durasiParts = explode(':', $durasi);
                    $hours = (int) $durasiParts[0];
                    $minutes = (int) $durasiParts[1];
                    $seconds = (int) $durasiParts[2];

                    $waktuSelesai = $waktuMulai->copy()
                        ->addHours($hours)
                        ->addMinutes($minutes)
                        ->addSeconds($seconds);

                    return Carbon::now()->greaterThan($waktuSelesai);
                });

            $stoppedBillings = [];

            foreach ($expiredBillings as $billing) {
                // Calculate total cost based on the set duration
                $tarif = $billing->tarif_perjam;
                $durasi = $billing->durasi;

                // Convert duration to hours for calculation
                $durasiParts = explode(':', $durasi);
                $totalHours = (int) $durasiParts[0] + ((int) $durasiParts[1] / 60) + ((int) $durasiParts[2] / 3600);

                $totalBiaya = $tarif * $totalHours;

                // Round up to nearest 100
                $sisa = $totalBiaya % 100;
                if ($sisa !== 0) {
                    $totalBiaya = $totalBiaya + (100 - $sisa);
                }

                // Update billing to completed
                $billing->update([
                    'status' => 'selesai',
                    'total_biaya' => $totalBiaya,
                    'waktu_selesai' => Carbon::now(),
                ]);

                $stoppedBillings[] = [
                    'billing_id' => $billing->id,
                    'device_id' => $billing->espRelay->device_id,
                    'pin' => $billing->espRelay->pin,
                    'nama_pelanggan' => $billing->nama_pelanggan,
                    'total_biaya' => $totalBiaya,
                    'durasi' => $billing->durasi,
                ];
            }

            return response()->json([
                'success' => true,
                'message' => 'Expired billings checked',
                'expired_count' => count($stoppedBillings),
                'stopped_billings' => $stoppedBillings,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to check expired billings: ' . $e->getMessage(),
            ], 500);
        }
    }
}
